/*!
 * \file wsenumvariant.hpp
 * \brief A means to define a std::variant<T<e0>, T<e1>, ...> for all values e0, e1, ... of a WSenum
 *
 *	Copyright 2019 WSoptics GmbH
 */

#pragma once

#include <utility>
#include <variant>

#include <wsenum.hpp>

namespace wsenum {
	namespace detail {
		template <bool addMonostate>
		struct VariantImplImpl;

		template <>
		struct VariantImplImpl<true> {
			template <typename E, template <E> typename T, UnderlyingType... index>
			static constexpr auto apply(std::integer_sequence<UnderlyingType, index...>) -> std::variant<std::monostate, T<static_cast<E>(index)>...>;
		};
		template <>
		struct VariantImplImpl<false> {
			template <typename E, template <E> typename T, UnderlyingType... index>
			static constexpr auto apply(std::integer_sequence<UnderlyingType, index...>) -> std::variant<T<static_cast<E>(index)>...> {
				static_assert(!std::is_same_v<T<static_cast<E>(0)>, std::monostate>,
				              "T<0> must not be std::monostate.  Use the boolean flag for adding a monostate to wsenum::Variant instead");
			}
		};

		template <typename E, template <E> typename T, bool addMonostate>
		struct VariantImpl {
			using Seq = std::make_integer_sequence<UnderlyingType, enumSize(E{})>;
			using type = decltype(VariantImplImpl<addMonostate>::template apply<E, T>(Seq{}));
			using EnumType = E;
		};

		template <auto e, typename V>
		constexpr std::size_t index() {
			// Depending on whether the first element is std::monostate or not, we have to be off by one.
			if constexpr (std::is_same_v<std::monostate, std::variant_alternative_t<0, V>>) {
				return static_cast<std::size_t>(e) + 1;
			} else {
				return static_cast<std::size_t>(e);
			}
		}
	} // namespace detail

	//! Define a variant over T<e> for a template T and all entries e of a given WSenum
	/*!
	 * For a WSenum E containing e_i for i < n, this creates std::variant<T<e_0>, T<e_1>, ..., T<e_n-1>> or std::variant<std::monostate, T<e_0>, T<e_1>, ..., T<e_n-1>>, depending on the value of
	 * addMonostate.
	 *
	 * \tparam T The template to instantiate with each possible value of E to create the std::variant with
	 * \tparam E The WSenum whose values to use
	 * \tparam addMonostate When true, std::monostate will be added as the first type to the variant;  defaults to false
	 */
	template <typename E, template <E> typename T, bool addMonostate = false>
	using Variant = typename detail::VariantImpl<E, T, addMonostate>::type;

	//! Return true if a wsenumvariant v contains a value of type e
	template <auto e, typename V>
	constexpr std::enable_if_t<isWSenum(e), bool> matches(V const & v) {
		// We have no chance to use std::holds_alternative since we cannot re-construct the template template parameter T used to create V.
		// So we use std::variant::index instead.
		return v.index() == detail::index<e, V>();
	}

	template <typename E, typename V>
	constexpr std::enable_if_t<isWSenum(E{}), E> index(V const & v) {
		static_assert(!std::is_same_v<std::variant_alternative_t<0, V>, std::monostate>, "index() is only supported for wsenum::Variant generated with addMonostate set to false");
		return static_cast<E>(v.index());
	}

	//! Return the content of a Variant by querying with the enum value
	///@{
	template <auto e, typename = std::enable_if_t<isWSenum(e)>, typename... Ts>
	constexpr decltype(auto) get(std::variant<Ts...> & g) {
		return std::get<detail::index<e, std::variant<Ts...>>()>(g);
	}
	template <auto e, typename = std::enable_if_t<isWSenum(e)>, typename... Ts>
	constexpr decltype(auto) get(std::variant<Ts...> && g) {
		return std::get<detail::index<e, std::variant<Ts...>>()>(std::move(g));
	}
	template <auto e, typename = std::enable_if_t<isWSenum(e)>, typename... Ts>
	constexpr decltype(auto) get(std::variant<Ts...> const & g) {
		return std::get<detail::index<e, std::variant<Ts...>>()>(g);
	}
	template <auto e, typename = std::enable_if_t<isWSenum(e)>, typename... Ts>
	constexpr decltype(auto) get(std::variant<Ts...> const && g) {
		return std::get<detail::index<e, std::variant<Ts...>>()>(g);
	}
	///@}
} // namespace wsenum
