/*!
 * \file wsenumtuple.hpp
 * \brief A means to define a std::tuple<T<e0>, T<e1>, ...> for all values e0, e1, ... of a WSenum
 *
 *	Copyright 2022 WSoptics GmbH
 */

#pragma once

#include <tuple>

#include <wsenum.hpp>

namespace wsenum {
	namespace detail {
		template <typename E, template <E> typename T, typename = std::enable_if_t<isWSenum(E{})>>
		struct ConstructTuple {
			template <UnderlyingType... es>
			static constexpr auto apply(std::integer_sequence<UnderlyingType, es...>) -> std::tuple<T<static_cast<E>(es)>...>;

			using Seq = std::make_integer_sequence<UnderlyingType, enumSize(E{})>;

			using type = decltype(apply(Seq{}));
		};
	} // namespace detail

	//! Create tuple where each element type is a template instantiation with an enum value
	/*!
	 * Example:
	 *
	 * ```
	 * WSENUM(Enum, (foo, bar));
	 *
	 * template <Enum>
	 * struct TupleElement;
	 *
	 * template <>
	 * struct TupleElement<Enum::foo>
	 * {
	 * };
	 *
	 * template <>
	 * struct TupleElement<Enum::bar>
	 * {
	 * };
	 *
	 * using TestTuple = Tuple<Enum, TupleElement>;
	 *
	 * static_assert(std::is_same_v<TestTuple, std::tuple<TupleElement<Enum::foo>, TupleElement<Enum::bar>>>);
	 * ```
	 */
	template <typename EnumT, template <EnumT e> typename TupleElement, typename = std::enable_if_t<isWSenum(EnumT{}), void>>
	using Tuple = typename detail::ConstructTuple<EnumT, TupleElement>::type;

	template <auto e, typename = std::enable_if_t<isWSenum(decltype(e){})>, typename... Ts>
	constexpr decltype(auto) get(std::tuple<Ts...> & tuple) {
		return std::get<static_cast<std::size_t>(e)>(tuple);
	}

	template <auto e, typename = std::enable_if_t<isWSenum(decltype(e){})>, typename... Ts>
	constexpr decltype(auto) get(std::tuple<Ts...> && tuple) {
		return std::get<static_cast<std::size_t>(e)>(std::move(tuple));
	}

	template <auto e, typename = std::enable_if_t<isWSenum(decltype(e){})>, typename... Ts>
	constexpr decltype(auto) get(std::tuple<Ts...> const & tuple) {
		return std::get<static_cast<std::size_t>(e)>(tuple);
	}

	template <auto e, typename = std::enable_if_t<isWSenum(decltype(e){})>, typename... Ts>
	constexpr decltype(auto) get(std::tuple<Ts...> const && tuple) {
		return std::get<static_cast<std::size_t>(e)>(tuple);
	}
} // namespace wsenum
