/*!
 * \file wsenum.hpp
 * \brief Macro for defining an enum with toString function giving entry names
 *
 *	Copyright 2016 - 2018 WSoptics GmbH
 */

#pragma once

#include <algorithm>
#include <array>
#include <optional>
#include <string_view>
#include <type_traits>
#include <utility>

#include <boost/preprocessor.hpp>

#include <cereal/access.hpp>

#include <constexpr_foreach.hpp>
#include <wsassert.hpp>
#include <wsdescriptor.hpp>
#include <wsexception.hpp>

template <typename T>
constexpr bool isWSenum(T const &) {
	return false;
}

namespace wsenum {
	//! Type for exception where a bad string was passed to toEnum()
	struct BadString : public WSexception {
		using WSexception::WSexception;
	};

	using UnderlyingType = int32_t;

	//! < Operator for WSenum
	template <typename T>
	std::enable_if_t<isWSenum(T{}), bool> operator<(T lhs, T rhs) {
		using U = std::underlying_type_t<T>;
		return static_cast<U>(lhs) < static_cast<U>(rhs);
	}
	//! <= Operator for WSenum
	template <typename T>
	std::enable_if_t<isWSenum(T{}), bool> operator<=(T lhs, T rhs) {
		using U = std::underlying_type_t<T>;
		return static_cast<U>(lhs) <= static_cast<U>(rhs);
	}
	//! > Operator for WSenum
	template <typename T>
	std::enable_if_t<isWSenum(T{}), bool> operator>(T lhs, T rhs) {
		using U = std::underlying_type_t<T>;
		return static_cast<U>(lhs) > static_cast<U>(rhs);
	}
	//! >= Operator for WSenum
	template <typename T>
	std::enable_if_t<isWSenum(T{}), bool> operator>=(T lhs, T rhs) {
		using U = std::underlying_type_t<T>;
		return static_cast<U>(lhs) >= static_cast<U>(rhs);
	}

	template <typename T>
	using EnumRange = wsdescriptor::DescriptorArray<wsenum::UnderlyingType, T>;

	//! Construct a DescriptorArray that runs through an entire WSenum
	/*!
	 * This function serves as a means to create range based loops through entire WSenum's as such:
	 * for(const auto e : enumRange<WSenumType>()) { ... }
	 */
	template <typename T>
	constexpr auto enumRange() {
		return EnumRange<T>(enumSize(T{}));
	}

	//! An integral constant type for an enum value
	template <typename T, T t>
	using EnumType = std::integral_constant<T, t>;

	namespace detail {
		template <typename E, typename F>
		struct ForEachWrapper {
			F const & f;

			template <typename Index>
			constexpr void operator()(Index) const {
				f(EnumType<E, static_cast<E>(Index{}())>{});
			}
		};
	} // namespace detail

	//! Call a function object for each element of a WSenum seen as a compile-time constant
	/*!
	 * This function can be used as follows for a given WSenum `Enum`:
	 * \code{.cpp}
	 * auto const lambda = [](auto const e) {
	 *         // EnumType is EnumType<Enum, enumValue>.
	 *         using EnumType = decltype(e);
	 *         // ce is thus enumValue.
	 *         constexpr Enum ce = EnumType()().
	           WSassert(ce == e);
	 *         doSomething<ce>();
	 * };
	 * wsenum::foreach<Enum>(lambda);
	 * \endcode
	 *
	 * Note that it does not pass a parameter of type Enum, but a parameter of type EnumType<Enum, e>, where e is the value of Enum that is handled in each iteration.
	 */
	template <typename E, typename F>
	constexpr void foreach (F const & f) {
		auto const wrappedF = detail::ForEachWrapper<E, F>{f};
		wstd::constexpr_foreach<enumSize(E{})>(wrappedF);
	}

} // namespace wsenum

#define X_WSENUM_CASE(r, data, elem) \
	case data::elem: return BOOST_PP_STRINGIZE(elem);

#define X_WSENUM_IFCLAUSE(r, data, elem)                                                \
	if (BOOST_PP_SEQ_ELEM(1, data) == toString(BOOST_PP_SEQ_ELEM(0, data)::elem)) { \
		return BOOST_PP_SEQ_ELEM(0, data)::elem;                                \
	}

/*!
 * This macro can be used like this: WSENUM(ClassName, (enumEntry0, enumEntry1, ...))
 *
 * Note that boost preprocessor's tuples are limited to 64 elements!
 * If you require a larger enum, look at the WSENUM_SEQ macro below.
 *
 * Then toString(ClassName) is defined and returns "enumEntry0", "enumEntry1", ...
 *
 * In addition, a constexpr function enumSize(ClassName) is provided that returns the number of elements of the enum.
 *
 * Also, a templated function toEnum(std::string_view, Enum) is defined for turning a std::string_view into an enum entry.
 * It throws wsenum::BadString if there is no entry of the given name.  A non-throwing alternative is toEnumOpt(std::string_view, Enum).
 *
 * Note that it would be natural to write `toEnum<Enum>(stringvalue)`.
 * However, this does not allow ADL, so an unqualified call to `toEnum` would not work.
 * By making the template type argument deducible by specifying the second argument, we make the compiler search for `toEnum` in `Enum`'s namespace.
 */
#define WSENUM(name, enumerators)                                                                              \
	enum class name : wsenum::UnderlyingType { BOOST_PP_TUPLE_ENUM(enumerators) };                         \
	[[maybe_unused]] constexpr bool isWSenum(name const) {                                                 \
		return true;                                                                                   \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr ::std::string_view enumName(name) {                                  \
		return #name;                                                                                  \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr ::std::string_view toString(name v) {                                \
		switch (v) { BOOST_PP_SEQ_FOR_EACH(X_WSENUM_CASE, name, BOOST_PP_TUPLE_TO_SEQ(enumerators)) }  \
		return "";                                                                                     \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr wsenum::UnderlyingType enumSize(name) {                              \
		return BOOST_PP_TUPLE_SIZE(enumerators);                                                       \
	}                                                                                                      \
	/* NOLINTNEXTLINE(readability-function-cognitive-complexity) */                                        \
	[[maybe_unused]] inline constexpr name toEnum(::std::string_view s, name) {                            \
		BOOST_PP_SEQ_FOR_EACH(X_WSENUM_IFCLAUSE, (name) (s), BOOST_PP_TUPLE_TO_SEQ(enumerators))       \
		throw wsenum::BadString(::std::string("Bad string for enum conversion: ") + ::std::string{s}); \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr std::optional<name> toEnumOpt(::std::string_view s, name) {          \
		BOOST_PP_SEQ_FOR_EACH(X_WSENUM_IFCLAUSE, (name) (s), BOOST_PP_TUPLE_TO_SEQ(enumerators))       \
		return std::nullopt;                                                                           \
	}

//! Macro using a boost processor sequence instead of a tuple
/*!
 * Since boost preprocessor sequences allow up to 256 entries, this macro allows for larger enums than WSENUM.
 * It can be used as such:
 * WSENUM_SEQ(ClassName, (e0)(e1)(e2))
 */
#define WSENUM_SEQ(name, enumerators)                                                                          \
	enum class name : wsenum::UnderlyingType { BOOST_PP_SEQ_ENUM(enumerators) };                           \
	[[maybe_unused]] constexpr bool isWSenum(name const) {                                                 \
		return true;                                                                                   \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr ::std::string_view enumName(name) {                                  \
		return #name;                                                                                  \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr ::std::string_view toString(name v) {                                \
		switch (v) { BOOST_PP_SEQ_FOR_EACH(X_WSENUM_CASE, name, enumerators) }                         \
		return "";                                                                                     \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr wsenum::UnderlyingType enumSize(name) {                              \
		return BOOST_PP_SEQ_SIZE(enumerators);                                                         \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr name toEnum(::std::string_view s, name) {                            \
		BOOST_PP_SEQ_FOR_EACH(X_WSENUM_IFCLAUSE, (name) (s), enumerators)                              \
		throw wsenum::BadString(::std::string("Bad string for enum conversion: ") + ::std::string{s}); \
	}                                                                                                      \
	[[maybe_unused]] inline constexpr std::optional<name> toEnumOpt(::std::string_view s, name) {          \
		BOOST_PP_SEQ_FOR_EACH(X_WSENUM_IFCLAUSE, (name) (s), enumerators)                              \
		return std::nullopt;                                                                           \
	}

namespace wsenum {
	//! Implements simple map for enums.
	template <typename EnumT, typename ValueT>
	class Map {
	public:
		constexpr Map() = default;

		//! Construct this Map from a parameter pack of alternating parameters, even parameters being of type EnumT, odd ones of ValueT
		template <typename... Ts, typename = std::enable_if_t<sizeof...(Ts) % 2 == 0>>
		constexpr explicit Map(Ts &&... ts);

		//! const ref operator
		constexpr const auto & operator[](EnumT const enumValue) const {
			WSassert(static_cast<int>(enumValue) >= 0);
			WSassert(static_cast<std::size_t>(enumValue) < values_.size());
			return values_[static_cast<std::size_t>(enumValue)];
		}

		//! ref operator
		constexpr auto & operator[](EnumT const enumValue) {
			WSassert(static_cast<int>(enumValue) >= 0);
			WSassert(static_cast<std::size_t>(enumValue) < values_.size());
			return values_[static_cast<std::size_t>(enumValue)];
		}

		constexpr auto size() const { return values_.size(); }

		//! \return Number of occurences of value
		constexpr auto count(ValueT const & value) const { return std::count(values_.cbegin(), values_.cend(), value); }

		//! \return True, if value has been found
		auto constexpr contains(ValueT const & value) const { return static_cast<bool>(key(value)); }

		//! \return Key for the first matching value; std::nullopt if value has not been found
		auto constexpr key(ValueT const & value) const;

		constexpr auto operator<=>(Map const &) const
		        requires(requires(ValueT l, ValueT r) {
			                { l <=> r };
		                })
		= default;

	private:
		//! map
		std::array<ValueT, enumSize(EnumT{})> values_{};

		//! Set one value at a time
		template <typename... Ts>
		constexpr void setValues(EnumT e, ValueT v, Ts &&... ts);

		friend class cereal::access;
		//! serialization
		template <typename Archive>
		void serialize(Archive & ar, const uint32_t) {
			ar(values_);
		}
	};

	// Deduction guide for Map
	template <typename E, typename V, typename... Ts>
	Map(E, V, Ts...) -> Map<E, V>;

	namespace detail {
		//! Check if a passed parameters is different from all even parameters of a parameter pack
		template <typename T>
		constexpr bool differentFromAllEven(T) {
			return true;
		}

		template <typename T, typename T0, typename T1, typename... Ts>
		constexpr bool differentFromAllEven(T t, T0 t0, T1 const &, Ts const &... ts) {
			return t != t0 && differentFromAllEven(t, ts...);
		}

		// Check that all even indexed parameters are pairwise different
		constexpr bool evenPairwiseDifferent() {
			return true;
		}
		template <typename T0, typename T1, typename... Ts>
		constexpr bool evenPairwiseDifferent(T0 t0, T1 const &, Ts const &... ts) {
			return differentFromAllEven(t0, ts...) && evenPairwiseDifferent(ts...);
		}
	} // namespace detail

	template <typename EnumT, typename ValueT>
	template <typename... Ts, typename>
	constexpr Map<EnumT, ValueT>::Map(Ts &&... ts) {
		static_assert(sizeof...(ts) <= 2 * enumSize(EnumT{}), "It makes no sense to have more than 2 * enumSize() many arguments for Map::Map");
		WSassert(detail::evenPairwiseDifferent(ts...) && "Map::Map expects all its parameters of type Enum to be pairwise different");
		setValues(std::forward<Ts>(ts)...);
	}

	template <typename EnumT, typename ValueT>
	auto constexpr Map<EnumT, ValueT>::key(ValueT const & value) const {
		auto const iter = std::find(values_.cbegin(), values_.cend(), value);
		return iter == values_.cend() ? std::nullopt : std::optional<EnumT>{static_cast<EnumT>(std::distance(values_.cbegin(), iter))};
	}

	template <typename EnumT, typename ValueT>
	template <typename... Ts>
	constexpr void Map<EnumT, ValueT>::setValues(EnumT const e, ValueT v, Ts &&... ts) {
		values_[static_cast<std::size_t>(e)] = std::move(v);
		if constexpr (sizeof...(ts) != 0) {
			setValues(std::forward<Ts>(ts)...);
		}
	}

	//! Factory function for creating a complete Map (i.e. a Map that has an entry for every element of the WSenum)
	/*!
	 * This function has the additional benefit of making sure that no value is missing when creating a Map.
	 * So when creating constant maps that should not miss an entry, it is preferable to use this factory as opposed to normal construction.
	 */
	template <typename EnumT, typename ValueT, typename... Ts>
	constexpr auto createEnumMap(EnumT const e, ValueT && v, Ts &&... ts) {
		static_assert(sizeof...(ts) + 2 == 2 * enumSize(EnumT{}), "createEnumMap expects exactly 2 * enumSize(EnumT{}) many parameters");
		return Map<std::remove_cvref_t<EnumT>, std::decay_t<ValueT>>(e, std::forward<ValueT>(v), std::forward<Ts>(ts)...);
	}
} // namespace wsenum
