/*!
 * \file wsdescriptors.hpp
 * \brief WSoptics descriptor handling
 *
 *	Copyright 2018 - 2020 WSoptics GmbH
 */

#pragma once

#include <compare>
#include <cstdint>
#include <iterator>
#include <type_traits>

#include <cereal/specialize.hpp>

#include <wsassert.hpp>

namespace wsdescriptor {
	//! Underlying type for descriptors
	using DescriptorType = uint32_t;
	//! Create an array of the form [0, ..., size) of types D and provide begin() and end() suitable for range based loops
	template <typename D, typename CastTo = D>
	class DescriptorArray {
	public:
		DescriptorArray() = delete;
		constexpr explicit DescriptorArray(D size) noexcept(noexcept(D{size})) : size_{size} {}
		class DescriptorIterator {
		private:
			D d_{};

			static constexpr auto ut(std::false_type) { return typename D::UnderlyingType{}; }

			static constexpr auto ut(std::true_type) { return std::underlying_type_t<CastTo>{}; }

		public:
			using UnderlyingType = decltype(ut(typename std::is_enum<CastTo>::type{}));

			using difference_type = std::make_signed_t<UnderlyingType>;
			using value_type = CastTo;
			using pointer = CastTo *;
			using reference = CastTo &;
			using iterator_category = std::random_access_iterator_tag;

			DescriptorIterator() = default;
			constexpr explicit DescriptorIterator(D d) noexcept(noexcept(D{d})) : d_{d} {}
			constexpr CastTo operator*() const noexcept { return static_cast<CastTo>(d_); }
			constexpr DescriptorIterator & operator++() noexcept(noexcept(++d_)) {
				++d_;
				return *this;
			}
			// Required for std::input_or_output_iterator concept
			// NOLINTNEXTLINE(readability-const-return-type)
			[[nodiscard]] constexpr DescriptorIterator const operator++(int) & noexcept(noexcept(this->operator++())) {
				auto const old = *this;
				this->operator++();
				return old;
			}
			constexpr DescriptorIterator & operator--() noexcept(noexcept(--d_)) {
				WSassert(d_ != 0);
				--d_;
				return *this;
			}
			// Required for std::input_or_output_iterator concept
			// NOLINTNEXTLINE(readability-const-return-type)
			[[nodiscard]] constexpr DescriptorIterator const operator--(int) & noexcept(noexcept(this->operator--())) {
				auto const old = *this;
				this->operator--();
				return old;
			}
			constexpr DescriptorIterator & operator+=(difference_type n) {
				if constexpr (std::is_signed_v<UnderlyingType>) {
					d_ += static_cast<UnderlyingType>(n);
				} else {
					if (n >= 0) {
						d_ += static_cast<UnderlyingType>(n);
					} else {
						d_ -= static_cast<UnderlyingType>(-n);
					}
				}
				return *this;
			}
			constexpr DescriptorIterator & operator-=(difference_type n) {
				if constexpr (std::is_signed_v<UnderlyingType>) {
					d_ -= static_cast<UnderlyingType>(n);
				} else {
					if (n >= 0) {
						d_ -= static_cast<UnderlyingType>(n);
					} else {
						d_ += static_cast<UnderlyingType>(-n);
					}
				}
				return *this;
			}

			constexpr difference_type operator-(DescriptorIterator other) const noexcept(noexcept(d_ - other.d_)) {
				return static_cast<difference_type>(d_) - static_cast<difference_type>(other.d_);
			}
			constexpr DescriptorIterator operator+(difference_type n) const {
				auto tmp = *this;
				return tmp += n;
			}
			constexpr DescriptorIterator operator-(difference_type n) const {
				auto tmp = *this;
				return tmp -= n;
			}
			constexpr auto operator[](UnderlyingType n) const { return *(*this + n); }
			friend constexpr auto operator<=>(DescriptorIterator, DescriptorIterator) = default;
		};
		constexpr DescriptorIterator begin() const noexcept(noexcept(DescriptorIterator{D{0}})) { return DescriptorIterator{D(0)}; }
		constexpr DescriptorIterator cbegin() const noexcept(noexcept(DescriptorIterator{D{0}})) { return DescriptorIterator{D(0)}; }
		constexpr DescriptorIterator end() const noexcept(noexcept(DescriptorIterator{D{0}})) { return DescriptorIterator{D(size_)}; }
		constexpr DescriptorIterator cend() const noexcept(noexcept(DescriptorIterator{D{0}})) { return DescriptorIterator{D(size_)}; }
		constexpr D size() const noexcept(noexcept(D(std::declval<D>()))) { return size_; }
		constexpr bool empty() const noexcept(noexcept(D{} == D{})) { return size_ == D{0}; }
		constexpr auto operator[](typename DescriptorIterator::UnderlyingType n) const { return begin()[n]; }

	private:
		D size_;
	};

	constexpr DescriptorType invalidDescriptor{static_cast<DescriptorType>(-1)};
} // namespace wsdescriptor
