/*!
 * \file constexpr_foreach.hpp
 * \brief Compile time for loop
 *
 *	Copyright 2019 - 2020 WSoptics GmbH
 */

#pragma once

#include <type_traits>
#include <utility>

namespace wstd {
	namespace detail {
		template <typename F, typename IndexType, IndexType... index>
		constexpr void apply(F const & f, std::integer_sequence<IndexType, index...>) {
			(f(std::integral_constant<IndexType, index>{}), ...);
		}
	} // namespace detail

	//! Calls object with a range of std::integral_constant:s
	/*
	 * \tparam size Defines the range of integral constants that will be passed to f
	 * \tparam F Type of the callable
	 * \param[in] f The callable
	 *
	 * The callable will be called with instances of std::integral_constant with instances of the range [0, size).
	 * The std::integral_constant's underlying type is deduced from size.
	 *
	 * Example: Replacement for boost::fusion::for_each when iterating over a boost fusion struct
	 *
	 * \code{.cpp}
	 * template <typename T>
	 * void printMembers(T const & t)
	 * {
	 * 	auto const printMember = [&t](auto const index) {
	 * 		std::cout << boost::fusion::at_c<index()>(t) << '\n';
	 * 	}
	 * 	wstd::constexpr_foreach<boost::fusion::result_of::size<T>::value>(printMember);
	 * }
	 *
	 * struct S
	 * {
	 * 	int i = 0;
	 * 	double d = 0.;
	 * };
	 * BOOST_FUSION_ADAPT_STRUCT(S)
	 *
	 * printMembers(S{42, 1.2});
	 * \endcode
	 */
	template <auto size, typename F>
	constexpr void constexpr_foreach(F const & f) {
		using IndexType = std::remove_cvref_t<decltype(size)>;
		using Seq = std::make_integer_sequence<IndexType, size>;
		detail::apply<F, IndexType>(f, Seq{});
	}
} // namespace wstd
